Option Strict Off
Option Explicit On
Imports System.IO
Imports System.Windows
Imports System.Text
Friend Class frmMain
    Inherits System.Windows.Forms.Form
    '
    ' File:       frmMain.frm
    '
    ' Purpose:    Simple program that demonstrates how to talk to the CAN232 in VB6
    '
    ' Version:    Created 1.0.0, 7:th of April 2003
    '                     2.0.0, 18 of August 2008   Howard L Honig, Honey Software LLC
    '
    ' Authors:
    '             Original Author
    '             Lars Wictorsson
    '             LAWICEL / SWEDEN
    '             http://www.lawicel.com/can232/   lars@lawicel.se
    '
    '             Modified by
    '             Howard Honig
    '             Honey Software LLC
    '             Corvallis, Oregon USA
    '             
    '
    ' Copyright:  The copyright to the computer program(s) herein is the
    '             property of LAWICEL HB, Sweden. The program(s) may be used
    '             and/or copied only with the written permission of LAWICEL HB
    '             in accordance with the terms and conditions stipulated in
    '             the agreement/contract under which the program(s) have been
    '             supplied. FMV also owns this code and can do what they want
    '             with it.
    '
    ' Manual:     Simply start program, it tries to open COM1 at 57600baud which
    '             is the default COM port and CAN232 RS232 Speed. If it fails, it
    '             will inform you that and you can choose a new COM port from the
    '             menu. Each time you change the COMport or Speed, the COMport closes
    '             and re-opens at the new speed (or port). Before sending or polling
    '             CAN frames, you need to Open the CAN232 (2 commands is processed
    '             by pushing "Open", the Sn[CR] and the O[CR] commands). If you do
    '             not do this, it will send a result back of [BELL]. Same happens if
    '             You try to close CAN232 if it isn't open, or open if it is Opened etc.
    '
    ' TODO:       - This program is rather simple and right now there is only
    '               one CAN speed defined and that is 125K, this will be changed soon.
    '             - Timeout from COMport (i.e. no respons from CAN232) shall be added.
    '
    ' History:    2003-04-07  1.0.0   Created (LWI)
    '             2008-08-17  2.0.0   Modified by H. Honig, Honey Software LLC

    Private Sub frmMain_Load(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles MyBase.Load
        Dim sCanSpeed As String
        iFlagACK = 0
        ComPortNo = GetSetting(Application.ProductName, "Com", "PortNumber", "1")
        SetPortMenu(ComPortNo)
        ComPortSpeed = GetSetting(Application.ProductName, "Com", "PortSpeed", "57600")
        SetPortSpeedMenu(ComPortSpeed)
        sCanSpeed = GetSetting(Application.ProductName, "Can", "Speed", "4")
        CANSpeed = sCanSpeed
        SetCanSpeedMnu(CANSpeed)
        ' OpenComPort()   Wait till command is sent to open
        TB_MaxLines.Text = GetSetting(Application.ProductName, "List", "MaxLines", 50)
        CB_AutoScrollDown.Checked = GetSetting(Application.ProductName, "List", "AutoScroll", 1)
        cbPrintAscii.Checked = GetSetting(Application.ProductName, "List", "PrintChars", 1)

        ' initialize status
        lbStatus_ComOpen.BackColor = Color.Red
        lbStatus_CanError.BackColor = Color.Red
        ' set filter values
        m_nFilter_PidLower = 0
        m_nFilter_PidUpper = &H3FFFFFFF
        m_bAutoPollSet = False              'Assume Auto Poll is off
        m_bPausePrinting = False
        m_CANPortStatus = CANPORTSTATUS.CPS_UNKNOWN

        'Window Pos
        'Move window to where the last point was.  If the window doesn't fit on the
        'monitor, then ignore the last coords
        Dim W_y As Short = GetSetting(Application.ProductName, "Window", "StartPos_Y", 40)
        Dim szPM As Size = SystemInformation.PrimaryMonitorMaximizedWindowSize()
        If (W_y > 0 And (W_y + Me.Height()) < szPM.Height()) Then
            Me.Top = W_y
        End If

        Dim W_x As Short = GetSetting(Application.ProductName, "Window", "StartPos_X", 50)
        If (W_x > 0 And (W_x + Me.Width()) < szPM.Width()) Then
            Me.Left = W_x
        End If

        uSendTimer = -1
        m_sTextResultBuffer = ""
        m_sSaveBuffer = ""
    End Sub
    Private Sub OnPaints(ByVal sender As System.Object, ByVal e As System.Windows.Forms.PaintEventArgs) Handles MyBase.Paint
        If (m_sSaveBuffer.Length > 0) Then
            Update_TxtResult()
        End If
    End Sub

    Private Sub frmMain_FormClosed(ByVal eventSender As System.Object, ByVal eventArgs As System.Windows.Forms.FormClosedEventArgs) Handles Me.FormClosed

        If (m_CANPortStatus = CANPORTSTATUS.CPS_OPEN) Then
            SendCommand("C")              'close can
        End If
        If (ComPortStatus = 1) Then
            SerialPort1.Close()
        End If

        SaveSetting(Application.ProductName, "Com", "PortNumber", ComPortNo.ToString)
        SaveSetting(Application.ProductName, "Com", "PortSpeed", ComPortSpeed)
        SaveSetting(Application.ProductName, "Can", "Speed", CANSpeed.ToString)
        SaveSetting(Application.ProductName, "List", "MaxLines", TB_MaxLines.Text)
        SaveSetting(Application.ProductName, "List", "AutoScroll", CB_AutoScrollDown.Checked)
        SaveSetting(Application.ProductName, "List", "PrintChars", cbPrintAscii.Checked)
        'Save window rectangle coordinates to be restored later
        SaveSetting(Application.ProductName, "Window", "StartPos_Y", Me.Top())
        SaveSetting(Application.ProductName, "Window", "StartPos_X", Me.Left())

    End Sub
    Private Sub Command1_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles btnGetVersion.Click
        Dim iResult As Short
        iResult = SendCommand("V")
    End Sub

    Private Sub ComSerNum_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnComSerNum.Click
        Dim iResult As Short
        iResult = SendCommand("N")
    End Sub

    'CAN Port Set speed and open CAN PORT
    Private Sub btnSetSpeed_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles btnSetSpeed_Open.Click
        Dim iResult As Short
        Dim sCmd As String
        sCmd = ""
        If (CANSpeed <= 8) Then
            sCmd = "S" & VB6.Format(CANSpeed, "0")
        ElseIf (CANSpeed = 83) Then
            sCmd = "s" & "051C"             '0B1C for 83Kpbs (really 83.333)
        End If

        iResult = SendCommand(sCmd)
        If (iResult = 1) Then            'Error
            m_CANPortStatus = CANPORTSTATUS.CPS_OPEN
            iResult = SendCommand("O")
        ElseIf (ComPortStatus = 1) Then      ' If false Serial Port failed to open
            MessageBox.Show("Error in opening CAN.  Possibly CAN port already opened.")
        End If

        If (iResult = 1) Then
            btnAutoPollOn.Enabled = False
            btnAutoPollOff.Enabled = False
            btnSetSpeed_Open.Enabled = False
            btnTimeStampOff.Enabled = False
            btnTimeStampOn.Enabled = False
            btnCloseCan.Enabled = True
        End If

    End Sub

    Private Sub btnCloseCan_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles btnCloseCan.Click
        Dim iResult As Short
        iResult = SendCommand("C")
        If (iResult <> 1) Then            'Error
            MessageBox.Show("Error in closing CAN.  Possibly CAN port already closed.")
        End If

        m_CANPortStatus = CANPORTSTATUS.CPS_CLOSED

        btnAutoPollOn.Enabled = True
        btnAutoPollOff.Enabled = True
        btnSetSpeed_Open.Enabled = True
        btnTimeStampOff.Enabled = True
        btnTimeStampOn.Enabled = True
        btnCloseCan.Enabled = False

    End Sub

    Private Sub btnPollOne_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles btnPollOne.Click
        Dim iResult As Short
        If (m_CANPortStatus <> CANPORTSTATUS.CPS_OPEN) Then
            MessageBox.Show("Can Port Closed", "CAN232HSW", MessageBoxButtons.OK)
            Exit Sub
        End If

        iResult = SendCommand("P")
        If ((iResult = 1) And (m_bAutoPollSet = 1)) Then
            m_bAutoPollSet = False
            m_CANPortStatus = CANPORTSTATUS.CPS_OPEN
        ElseIf ((iResult <> 1) And (ComPortStatus = 1)) Then         'Don't print message if ComPort is Closed
            MessageBox.Show("Error in CAN port. Possibly Auto Poll Mode is on")
        End If
    End Sub

    Private Sub btnPollAll_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles btnPollAll.Click
        Dim iResult As Short
        If (m_CANPortStatus <> CANPORTSTATUS.CPS_OPEN) Then
            MessageBox.Show("Can Port Closed", "CAN232HSW", MessageBoxButtons.OK)
            Exit Sub
        End If
        iResult = SendCommand("A")

        If ((iResult = 1) And (m_bAutoPollSet = True)) Then
            m_bAutoPollSet = False
            m_CANPortStatus = CANPORTSTATUS.CPS_OPEN
        ElseIf ((iResult <> 1) And (ComPortStatus = 1)) Then         'Don't print message if ComPort is Closed
            MessageBox.Show("Error in CAN port. Possibly Auto Poll Mode is on")
        End If
    End Sub


    Private Sub btnAutoPollOn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnAutoPollOn.Click
        Dim iResult As Short

        'Check to see if ComPort if Closed. If not offer to close it.
        If (m_CANPortStatus = CANPORTSTATUS.CPS_OPEN) Then
            If (MessageBox.Show("Can Port open, Close?", "CAN232HSW", MessageBoxButtons.OKCancel) = DialogResult.OK) Then
                iResult = SendCommand("C")
            End If
        End If
        iResult = SendCommand("X1")
        If (iResult = 1) Then
            m_bAutoPollSet = True
            m_CANPortStatus = CANPORTSTATUS.CPS_OPEN
        ElseIf ((iResult <> 1) And (ComPortStatus = 1)) Then         'Don't print message if ComPort is Closed
            MessageBox.Show("Error in CAN port. Possibly already in Auto Poll mode or CAN Port is closed")
        End If
    End Sub
    Private Sub btnAutoPollOff_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnAutoPollOff.Click
        Dim iResult As Short
        If (m_CANPortStatus = CANPORTSTATUS.CPS_OPEN) Then
            If (MessageBox.Show("Can Port open, Close?", "CAN232HSW", MessageBoxButtons.OKCancel) = DialogResult.OK) Then
                iResult = SendCommand("C")
            End If
        End If
        iResult = SendCommand("X0")
        If (iResult = 1) Then
            m_bAutoPollSet = False
            m_CANPortStatus = CANPORTSTATUS.CPS_OPEN
        ElseIf ((iResult <> 1) And (ComPortStatus = 1)) Then         'Don't print message if ComPort is Closed
            MessageBox.Show("Error in CAN port. Possibly already Auto Poll mode is off or CAN Port is closed")
        End If
    End Sub


    ' Handler for Time Stamp buttons on/off
    Private Sub btnTimeStampOn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnTimeStampOn.Click
        Dim iResult As Short
        iResult = 1
        If (m_CANPortStatus = CANPORTSTATUS.CPS_OPEN) Then
            If (MessageBox.Show("Can Port open, Close?", "CAN232HSW", MessageBoxButtons.OKCancel) = DialogResult.OK) Then
                iResult = SendCommand("C")
            End If
        End If

        If (iResult = 1) Then
            iResult = SendCommand("Z1")
        End If
    End Sub
    Private Sub btnTimeStampOff_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnTimeStampOff.Click
        Dim iResult As Short
        iResult = 1
        If (m_CANPortStatus = CANPORTSTATUS.CPS_OPEN) Then
            If (MessageBox.Show("Can Port open, Close?", "CAN232HSW", MessageBoxButtons.OKCancel) = DialogResult.OK) Then
                iResult = SendCommand("C")
            End If
        End If
        If (iResult = 1) Then
            iResult = SendCommand("Z0")
        End If
    End Sub


    ' Handles send data
    Private Sub btnSendData_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Command6.Click
        Dim iResult, iTmp As Short
        Dim sCmd As String
        'make sure id is 3 chars in length
        While (txtID.Text.Length < 3)
            txtID.Text = "0" & txtID.Text
        End While
        'and no more then 3 chars, really can't happen because the control is set to max 3 chars
        While (txtID.Text.Length > 3)
            txtID.Text = txtID.Text.Remove(0, 1)
        End While
        sCmd = "t" & txtID.Text & txtDLC.Text
        If (Val(txtDLC.Text) > 0) Then
            For iTmp = 1 To Val(txtDLC.Text)
                sCmd = sCmd & txtByte(iTmp - 1).Text
            Next
        End If
        iResult = SendCommand(sCmd)
    End Sub

    Private Sub btnSendMulti_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnSendMulti.Click
        SendMultiLines.Show()
    End Sub


    Public Sub mnu115200_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnu115200.Click
        ComPortSpeed = "115200"
        SetPortSpeedMenu(ComPortSpeed)
        OpenComPort()
    End Sub

    Public Sub mnu19200_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnu19200.Click
        ComPortSpeed = "19200"
        SetPortSpeedMenu(ComPortSpeed)
        OpenComPort()
    End Sub

    Public Sub mnu57600_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnu57600.Click
        ComPortSpeed = "57600"
        SetPortSpeedMenu(ComPortSpeed)
        OpenComPort()
    End Sub

    Public Sub mnu9600_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnu9600.Click
        ComPortSpeed = "9600"
        SetPortSpeedMenu(ComPortSpeed)
        OpenComPort()
    End Sub

    '****************************************************************************************************
    '        Com Set Port number functions and Methods
    '
    Public Sub mnuCOM1_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnuCOM1.Click
        SetPortMenu(1)
        ComPortNo = 1
        OpenComPort()
    End Sub

    Public Sub mnuCOM2_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnuCOM2.Click
        SetPortMenu(2)
        ComPortNo = 2
        OpenComPort()
    End Sub

    Private Sub mnuOtherComPortNum_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnuOtherComPortNum.Click
        Dim nSaveCom As Short

        nSaveCom = ComPortNo                   'save if needed to restore

        SetPortMenu(3)
        SelectComPort.SetComPort(ComPortNo)
        SelectComPort.ShowDialog()
        If (SelectComPort.DialogResult = Windows.Forms.DialogResult.OK) Then
            ComPortNo = SelectComPort.GetComPort()
            OpenComPort()
            If (SerialPort1.IsOpen = False) Then
                ComPortNo = nSaveCom
                SetPortMenu(ComPortNo)
            End If
        End If
    End Sub
    Private Sub SetPortMenu(ByVal pn As Short)
        'Com1 and Com2 left in on menu for convenience and legacy
        mnuCOM1.Checked = False
        mnuCOM2.Checked = False
        mnuOtherComPortNum.Checked = False
        If (pn = 1) Then
            mnuCOM1.Checked = True
        ElseIf (pn = 2) Then
            mnuCOM2.Checked = True
        ElseIf (pn >= 3) Then
            mnuOtherComPortNum.Checked = True
        End If
    End Sub

    Private Sub OpenCOMPortToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles OpenCOMPortToolStripMenuItem.Click
        OpenComPort()
    End Sub
    Private Sub OnMnuDropDownOpening(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnuSettings.DropDownOpening
        mnuOtherComPortNum.Text = "Other Com Port (Now COM" + ComPortNo.ToString + ")"
    End Sub
    '************************************************************************************************
    '    Other Handlers
    '
    Public Sub mnuExit_Click(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles mnuExit.Click
        Close()
    End Sub

    '
    '
    '***********************************************************************************************************
    ' Receive and handle the messages from the CAN
    '***********************************************************************************************************
    'Note: This method may be entered before the CAN232HSW has completely sent out the full line.
    ' Store the line until a <cr> has been received
    Dim m_sSaveBuffer As String


    Private Sub SerialPort1_DataReceived(ByVal sender As System.Object, ByVal e As System.IO.Ports.SerialDataReceivedEventArgs) Handles SerialPort1.DataReceived
        Dim i As Integer

        i = SerialPort1.BytesToRead             'Must be prior to the If statement for some dumb reason
        If (i > 0) Then
            m_sSaveBuffer = m_sSaveBuffer + SerialPort1.ReadExisting
            Me.Invalidate()
        End If

    End Sub

    Public Sub Update_TxtResult()
        Dim i As Short
        'Dim iTmp2, iTmp1, iTmp3 As Short
        Dim nLineLen As Integer
        Dim sTimeStamp As String
        Dim sOut, sParseString, sTemp1, sTemp2, ErrMsg As String

        ErrMsg = ""
        ' first check if there is a cr or bell in savebuffer
        Dim iCrPosition As Integer = m_sSaveBuffer.IndexOf(Chr(13))
        Dim iBellPosition As Integer = m_sSaveBuffer.IndexOf(Chr(7))
        sOut = ""
        While (iCrPosition >= 0) Or (iBellPosition >= 0)

            'Move the next response to the sParseString
            sParseString = ""
            sOut = ""
            If (iCrPosition >= 0) And ((iBellPosition = -1) Or (iBellPosition > iCrPosition)) Then
                sParseString = Mid(m_sSaveBuffer, 1, iCrPosition + 1)
                m_sSaveBuffer = m_sSaveBuffer.Remove(0, iCrPosition + 1)
            Else
                sParseString = Mid(m_sSaveBuffer, 1, iBellPosition + 1)
                m_sSaveBuffer = m_sSaveBuffer.Remove(0, iBellPosition + 1)
            End If


            Select Case Mid(sParseString, 1, 1)
                Case Chr(7)
                    'sOut = "[Bell]"
                    System.Diagnostics.Debug.Write("[BELL]")
                    iFlagACK = 2
                    lbStatus_CanError.BackColor = Color.Red
                Case Chr(13)
                    'sOut = "[CR]"
                    lbStatus_CanError.BackColor = Color.Green

                    iFlagACK = 1
                Case "t"
                    sTimeStamp = ""
                    nLineLen = sParseString.LastIndexOf(Chr(13))       'length of line
                    sTemp1 = Mid(sParseString, 1, nLineLen)
                    sOut = Mid(sTemp1, 2, 3) + " "       'pid
                    Dim tt As UInt32 = HexToDec(Mid(sTemp1, 2, 3))
                    Dim sAsciiOut As String = ""
                    If (m_nFilter_PidLower <= tt) And (m_nFilter_PidUpper >= tt) Then
                        Dim nByteCount As Integer = Mid(sTemp1, 5, 1)
                        sOut = sOut + nByteCount.ToString + " "
                        If (nByteCount > 0) Then
                            For i = 1 To nByteCount Step 1
                                sTemp2 = Mid(sTemp1, 6 + ((i - 1) * 2), 2)
                                sOut = sOut + sTemp2 + " "
                                'got a byte, figure out char, if on
                                sTemp2 = Chr(HexToDec(sTemp2))
                                If (sTemp2 <= Chr(&H1F) Or sTemp2 >= Chr(&H7F)) Then
                                    sTemp2 = "?"
                                End If
                                sAsciiOut = sAsciiOut & sTemp2
                            Next i
                        End If


                        'If requested show the ascii
                        If (cbPrintAscii.Checked() = True) Then
                            sOut = sOut & ":" & sAsciiOut.ToString
                        End If

                        'If timestamp is available add it to the end of the line
                        'Moved to the end to allow multiline send to ignore the time stamp
                        'if a cut/paste was done by user from the txtResult window
                        'assume timestamp if byte at the end of the frame is not <cr>
                        If (nLineLen = (nByteCount * 2 + 5 + 4)) Then
                            sTimeStamp = Mid(sTemp1, nByteCount * 2 + 6, 4)
                            sTimeStamp = Format(HexToDec(sTimeStamp), "000## ")
                            If (sTimeStamp.Length > 0) Then
                                sOut = sOut & " " & sTimeStamp
                            End If
                        End If

                    Else
                        sOut = ""
                    End If

                    iFlagACK = 1

                Case "z"                                        'Auto Poll mode is enabled
                    iFlagACK = 1
                    lbStatus_CanError.BackColor = Color.Green

                Case "Z"                                        'Auto Poll mode is enabled and received Z[cr]
                    iFlagACK = 1
                    lbStatus_CanError.BackColor = Color.Green

                Case Else
                    sOut = sParseString
                    iFlagACK = 1
            End Select

            iCrPosition = m_sSaveBuffer.IndexOf(Chr(13))
            iBellPosition = m_sSaveBuffer.IndexOf(Chr(7))

            'Add to the buffer
            If ((Len(sOut) > 0) And (m_bPausePrinting = False)) Then
                m_sTextResultBuffer = m_sTextResultBuffer & sOut & Chr(13) & Chr(10)
            End If

        End While

        If (ErrMsg.Length > 0) Then
            Dim sMess As String = "Error in COM port communication:" & ErrMsg
            MessageBox.Show(ErrMsg)
        End If

        If (Len(m_sTextResultBuffer) > 0) Then
            txtResult.Text = m_sTextResultBuffer
            'Set green ok flag if command received properly
            If (iFlagACK = 1) Then
                lbStatus_CanError.BackColor = Color.Green
            End If
        End If

        Dim strArray() As String
        strArray = txtResult.Lines
        '  Auto scroll and cut feature
        If (m_bPausePrinting = False) Then
            i = TB_MaxLines.Text
            If ((i <> 0) And (strArray.GetUpperBound(0) >= TB_MaxLines.Text)) Then
                txtResult.SelectAll()
                txtResult.Cut()
                m_sTextResultBuffer = ""
            ElseIf (txtResult.TextLength > 0) And (CB_AutoScrollDown.Checked = True) Then
                txtResult.SelectionStart = txtResult.TextLength - 1
                txtResult.SelectionLength = 1
                txtResult.ScrollToCaret()
            End If
        End If
    End Sub


    Private Sub SerialPort1_ErrorReceived(ByVal sender As System.Object, ByVal e As System.IO.Ports.SerialErrorReceivedEventArgs) Handles SerialPort1.ErrorReceived
        'The following are handlers of error messages. They should NOT affect CAN232
        Dim ErrMsg As String
        Select Case e
            'Case RXOver
            'ErrMsg = "RX overrun"
            ' These next three timeout values are no longer detected in Win32
            'Case comEvCTS
            'ErrMsg = "CTS changed"              ' CD (RLSD) Timeout.
            'Case comEvRing
            'ErrMsg = "You Rang"                 ' CTS Timeout.
            'Case comEvDSR
            'ErrMsg = "DSR Changed"                 ' CTS Timeout.
        End Select
    End Sub

    'This Subroutine was added as a watchdog timer for the SendCommand function in modGlobal module
    Private Sub OnTimerSendStatus(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Timer1.Tick
        If (iFlagACK = 0) Then
            m_sSaveBuffer = ""
            iFlagACK = 2
            MessageBox.Show("Error in COM port communication")
        End If
    End Sub

    Private Sub btnClearResult_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnClearResult.Click
        txtResult.Text = ""
        m_sTextResultBuffer = ""
    End Sub
    Private Sub btnPauseResult_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles btnPauseResult.Click
        m_bPausePrinting = Not m_bPausePrinting
        If (m_bPausePrinting = True) Then
            btnPauseResult.BackColor = Color.Red
        Else
            btnPauseResult.BackColor = btnClearResult.BackColor
        End If
    End Sub
    Private Sub txtByte_Leave(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles txtByte.Leave
        Dim Index As Short = txtByte.GetIndex(eventSender)
        If (Len(txtByte(Index).Text) = 1) Then
            txtByte(Index).Text = "0" & txtByte(Index).Text
        End If
    End Sub

    Private Sub txtID_Leave(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles txtID.Leave
        If (Len(txtID.Text) = 1) Then
            txtID.Text = "0" & txtID.Text
        End If
        If (Len(txtID.Text) = 2) Then
            txtID.Text = "0" & txtID.Text
        End If
    End Sub

    Private Sub mnu10K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu10K.Click
        CANSpeed = 0
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu20K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu20K.Click
        CANSpeed = 1
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu50K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu50K.Click
        CANSpeed = 2
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub Mnu83K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Mnu83K.Click
        CANSpeed = 83
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu100K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu100K.Click
        CANSpeed = 3
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu125K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu125K.Click
        CANSpeed = 4
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu250K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu250K.Click
        CANSpeed = 5
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu500K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu500K.Click
        CANSpeed = 6
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu800K_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu800K.Click
        CANSpeed = 7
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub mnu1M_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnu1M.Click
        CANSpeed = 8
        SetCanSpeedMnu(CANSpeed)
    End Sub
    Private Sub SetCanSpeedMnu(ByVal cs As Short)
        mnu10K.Checked = False
        mnu20K.Checked = False
        mnu50K.Checked = False
        Mnu83K.Checked = False
        mnu100K.Checked = False
        mnu125K.Checked = False
        mnu250K.Checked = False
        mnu500K.Checked = False
        mnu800K.Checked = False
        mnu1M.Checked = False
        If (cs = 0) Then
            mnu10K.Checked = True
        ElseIf (cs = 1) Then
            mnu20K.Checked = True
        ElseIf (cs = 2) Then
            mnu50K.Checked = True
        ElseIf (cs = 3) Then
            mnu100K.Checked = True
        ElseIf (cs = 4) Then
            mnu125K.Checked = True
        ElseIf (cs = 5) Then
            mnu250K.Checked = True
        ElseIf (cs = 6) Then
            mnu500K.Checked = True
        ElseIf (cs = 7) Then
            mnu800K.Checked = True
        ElseIf (cs = 8) Then
            mnu1M.Checked = True
        ElseIf (cs = 83) Then
            Mnu83K.Checked = True
        End If
    End Sub


    Private Sub SetPortSpeedMenu(ByVal br As String)
        mnu9600.Checked = False
        mnu19200.Checked = False
        mnu57600.Checked = False
        mnu115200.Checked = False
        If (br = "9600") Then
            mnu9600.Checked = True
        ElseIf (br = "19200") Then
            mnu19200.Checked = True
        ElseIf (br = "57600") Then
            mnu57600.Checked = True
        ElseIf (br = "115200") Then
            mnu115200.Checked = True
        End If

    End Sub

    Private Sub FilterToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles FilterToolStripMenuItem.Click
        ' Brings up the filter for the CAN Pid.  Initializes to no filter (range 0 - #ffffff)
        Dim spf As Object
        Dim sFilter_PidLower, sFilter_PidUpper As String
        sFilter_PidLower = ""
        sFilter_PidUpper = ""
        spf = SelectPidFilter

        spf.SetFilterValues(Hex(m_nFilter_PidLower), Hex(m_nFilter_PidUpper))
        spf.ShowDialog()
        If (spf.DialogResult = Windows.Forms.DialogResult.OK) Then
            spf.GetFilterValues(sFilter_PidLower, sFilter_PidUpper)
            m_nFilter_PidLower = HexToDec(sFilter_PidLower)
            m_nFilter_PidUpper = HexToDec(sFilter_PidUpper)
        End If
    End Sub

    Private Function HexToDec(ByVal sHex As String) As UInt32
        Dim txt As String = UCase$(Trim$(sHex))
        If (txt <> "") Then
            If (Microsoft.VisualBasic.Left$(txt, 2) <> "&H") Then txt = "&H" & txt
            HexToDec = CLng(txt)
        End If
    End Function
    '*******************************************************************************
    'File Stuff

    Private Sub mnuSave_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnuSave.Click
        WriteFile()
    End Sub
    Private Sub mnuSaveAsFile_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles mnuSaveAsFile.Click
        Dim oSD As New OpenFileDialog()
        oSD.FileName = m_sFileName
        oSD.AddExtension = True
        oSD.CheckFileExists = False
        oSD.DefaultExt = "txt"
        oSD.ValidateNames = True
        oSD.Filter = "Text Files (*.txt)|*.txt|All Files (*.*)|*.*"
        oSD.Title = "Select a file for results"
        If (oSD.ShowDialog() = DialogResult.OK) Then
            m_sFileName = oSD.FileName
            WriteFile()
        End If
    End Sub

    Private Sub WriteFile()
        Dim fs As StreamWriter

        If (m_sFileName = "") Then
            MessageBox.Show("Must specify filename")
            Exit Sub
        End If

        If File.Exists(m_sFileName) = False Then
            fs = File.CreateText(m_sFileName)
        Else
            fs = File.AppendText(m_sFileName)
        End If

        fs.Write("The date is: ")
        fs.WriteLine(DateTime.Now)
        fs.Write(txtResult.Text)
        fs.Flush()
        fs.Close()

    End Sub


    Private Sub AboutCAN232HSWToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles AboutCAN232HSWToolStripMenuItem.Click
        AboutBox1.Show()
    End Sub


End Class
